package com.rameses.rcp.common;

import com.rameses.rcp.annotations.*;
import com.rameses.osiris2.client.*;
import com.rameses.osiris2.common.*;
import com.rameses.util.Base64Cipher;

public class FileUploader {
    
    private def _entity;
    private def _files;
    private def _listeners;

    String connection; 
    String fileLocId;
    String fileDir;
    
    // If set to true, all files uploaded will be merged or appended 
    // to the current sys_file record 
    boolean append;

    public FileUploader( Map data ) {
        this._entity = data; 
        this._listeners = [];
        this._files = [];
    }
    
    public void setConnection( String name ) {
        this.connection = name; 
    }
    
    public void setFileLocId( String locid  ) {
        this.fileLocId = locid; 
    }
    

    public void addFile( java.io.File file ) {
        if ( file != null ) {
            _files << file;
        }
    }
    public void removeFile( java.io.File file ) {
        if ( file != null ) {
            _files.remove( file ); 
        }
    }
    public void removeFiles() {
        _files.clear(); 
    }

    
    public void addListener( FileUploadListener listener ) {
        if ( listener != null && !_listeners.contains( listener )) {
            _listeners << listener; 
        }
    }
    public void removeListener( FileUploadListener listener ) {
        if ( listener != null ) {
            _listeners.remove( listener ); 
        }
    }
    public void removeListeners() {
        _listeners.clear(); 
    }

    
    final def findLocationConf() {
        def fm = com.rameses.filemgmt.FileManager.instance;

        def locconf = null; 
        if ( fileLocId ) {
            locconf = fm.getLocation( connection, fileLocId ); 
            if ( !locconf ) throw new Exception("'"+ filelocid +"' fileloc record not found"); 
        } 
        else {
            locconf = fm.getDefaultLocation( connection ); 
            if ( !locconf ) throw new Exception('No active location config available'); 
        }
        return locconf; 
    }
    
    
    public void execute() {
        def fileLocConf = findLocationConf(); 
        if ( fileLocConf == null ) { 
            throw new Exception('Please provide a default location conf'); 
        }

        if ( !_files ) { 
            throw new Exception('Please provide at least 1 file to upload'); 
        }
        
        def fum = com.rameses.filemgmt.FileUploadManager.instance; 
        def tempdir = fum.getTempDir(); 

        def fileMapping = saveSysFile( fileLocConf );

        notifyOnStart();
        
        _entity.items.each{ item-> 
            item.immediate = true; 
            item.connection = connection;
            
            def file = fileMapping.get( item.objid ); 
            def folder = new java.io.File( tempdir, item.fileid ); 
            def fui = com.rameses.filemgmt.FileUploadItem.create( folder, item ); 

            def fileHandler = [
                onTransfer: { fuitm, filesize, bytesprocessed-> 
                    def data = fuitm.getConfigFile().copyData(); 

                    // remove unwanted properties
                    data.remove('fileid');
                    data.remove('state');
                    data.remove('immediate');
                    data.remove('caption');

                    // add new properties
                    data.filesize = filesize; 
                    data.bytestransferred = bytesprocessed; 

                    // DATA contains the following: 
                    // - objid, parentid, caption, source
                    // - filelocid, filetype, filename
                    // - filesize, bytestransferred
                    notifyOnProcess( data ); 
                    updateFileItemStat( data ); 
                },
                onCompleted: { fuitm-> 
                    notifyOnEnd(); 
                }
            ] as com.rameses.filemgmt.FileUploadManager.FileHandler; 

            fum.getFileHandlers().add( fui, fileHandler ); 
            fum.schedule( fui ); 
        }
    }
    
    
    def saveSysFile( fileLocConf ) {
        def fm = com.rameses.filemgmt.FileManager.instance;
        int fileIndexNo = 0; 
        def items = [];
        
        if ( !_entity.objid ) {
            _entity.objid = "FILE"+ new java.rmi.server.UID().toString();
        }

        def fum = com.rameses.filemgmt.FileUploadManager.instance; 
        def tempdir = fum.getTempDir(); 
        def fileTypeProvider = fm.fileTypeProvider; 

        def scaler = new ImageScaler();
        def base64 = new Base64Cipher(); 
        def fileMapping = [:];
        
        def createHash = {
            def str = ""+ new Object().hashCode(); 
            return str.replaceAll("-","");
        }
        
        _files.each{ file-> 
            int idx = file.getName().lastIndexOf('.'); 
            if ( idx <= 0 ) {
                throw new Exception("'"+ file.getName() +"' file not supported"); 
            }

            def sfiletype = file.getName().substring( idx + 1 );
            if ( !sfiletype ) {
                throw new Exception("'"+ file.getName() +"' file not supported"); 
            }
            
            fileIndexNo += 1; 
            def item = [:];
            def encstr = com.rameses.util.Encoder.MD5.encode( _entity.objid ); 
            item.objid = encstr + createHash().toString().padLeft(10, '0'); 
            item.parentid = _entity.objid;
            item.filesize = fum.helper.getFileSize( file ); 
            item.filelocid = fileLocConf.name; 
            item.filedir = fileDir;
            item.filetype = sfiletype;
            item.fileid = item.objid; 
            item.filename = file.name; 
            item.source = file.canonicalPath; 
            item.caption = file.name; 
            item.state = 'PENDING';
            
            def image = null; 
            def aaa = fileTypeProvider.getType( item.filetype );
            if ( aaa?.image && aaa.image.toString().matches('true|1')) {
                image = scaler.createThumbnail( file );  
            } 
            else {
                def icon = fm.getFileTypeIcon( item.filetype ); 
                if ( icon ) image = icon.image; 
            }
            
            if ( image ) {
                item.thumbnail = base64.encode((Object) scaler.getBytes( image )); 
            }

            items << item; 
            fileMapping.put( item.objid, file ); 
        }
            
        _entity.items = items;
        _entity.appendToFileId = ( this.append ? _entity.objid : null );

        def db = fm.getDbProvider(); 
        def o = db.create( _entity, connection ); 
        if ( o ) { 
            o.remove('items'); 
            _entity.putAll( o ); 
            _entity.items = items; 
        } 
        return fileMapping;
    }

    void updateFileItemStat( item ) { 
        def fm = com.rameses.filemgmt.FileManager.instance;
        def db = fm.getDbProvider(); 
        def svc = db.getService( connection );
        def param = [:]; 
        param.state = 'PROCESSING';
        param.bytestransferred = item.bytestransferred; 
        if ( param.bytestransferred >= param.filesize ) { 
            param.state = 'COMPLETED'; 
        } 
        svc.updateItemStat( item.objid, param ); 
    }
    
    void notifyOnStart() {
        _listeners.each{ o-> 
            try { 
                o.onStart(); 
            } catch(Throwable t) { 
                t.printStackTrace(); 
            } 
        }
    }
    void notifyOnProcess( Map param ) {
        _listeners.each{ o-> 
            try { 
                o.onProcess( param ); 
            } catch(Throwable t) { 
                t.printStackTrace(); 
            } 
        }
    }
    void notifyOnEnd() {
        _listeners.each{ o-> 
            try { 
                o.onEnd(); 
            } catch(Throwable t) { 
                t.printStackTrace(); 
            } 
        }
    }
}